unit IWCompMenu;

interface

uses
  {$IFDEF VSNET}
  System.ComponentModel, System.Drawing, System.Drawing.Design,IWNetClasses, IWNetComponent, System.Collections,
  {$ELSE}
    {$IFDEF Linux}QControls,{$ELSE}Controls,{$ENDIF}
    {$IFDEF Linux}Types, {$ELSE}Windows, {$ENDIF}
    {$IFDEF Linux}QMenus,{$ELSE}Menus,{$ENDIF}
    IWImageList,
  {$ENDIF}
  {$IFDEF Linux}QGraphics,{$ELSE}Graphics,{$ENDIF}
  Classes,
  IWAppForm, IWHTMLTag, IWControl, IWServerControllerBase,
  IWColor, IWFont, SWSystem, IWRenderContext, IWBaseHTMLInterfaces,
  IWBaseInterfaces;

type
  TIWMenu = class;

  TIWMenuOrientation = (iwOHorizontal, iwOVertical);

  {$IFDEF VSNET}
  [TypeConverter(typeof(ExpandableObjectConverter))]
  {$ENDIF}
  TIWMenuStyle = class(TPersistent)
  private
    FSelectedItemBGColor: TIWColor;
    FItemBGColor: TIWColor;
    FDisabledItemBGColor: TIWColor;
    FBGColor: TIWColor;
    FBorderColor : TIWColor;
    procedure SetDisabledItemBGColor(const Value: TIWColor);
    procedure SetItemBGColor(const Value: TIWColor);
    procedure SetSelectedItemBGColor(const Value: TIWColor);
    procedure SetBGColor(const Value: TIWColor);
    procedure SetBorderColor(const Value: TIWColor);
  protected
    FColor: TIWColor;
    FSelectedMenuItemFont: TIWFont;
    FMenuItemFont: TIWFont;
    FDisabledMenuItemFont : TIWFont;

    procedure SetMenuItemFont(const Value: TIWFont);
    procedure SetSelectedMenuItemFont(const Value: TIWFont);
    procedure SetDisabledMenuItemFont(const Value: TIWFont);
  protected
    FIWMenu: TIWMenu;
    FOrientation: TIWMenuOrientation;

    procedure SetOrientation(const Value: TIWMenuOrientation);
  public
    constructor Create(AIWMenu: TIWMenu);

    destructor Destroy; override;

    procedure Assign(Source: TPersistent); override;
  published
    property MenuItemFont: TIWFont read FMenuItemFont write SetMenuItemFont;
    property SelectedMenuItemFont: TIWFont read FSelectedMenuItemFont write SetSelectedMenuItemFont;
    property DisabledMenuItemFont: TIWFont read FDisabledMenuItemFont write SetDisabledMenuItemFont;

    property BGColor: TIWColor read FBGColor write SetBGColor;
    property ItemBGColor: TIWColor read FItemBGColor write SetItemBGColor;
    property SelectedItemBGColor: TIWColor read FSelectedItemBGColor write SetSelectedItemBGColor;
    property DisabledItemBGColor: TIWColor read FDisabledItemBGColor write SetDisabledItemBGColor;
    property BorderColor : TIWColor read FBorderColor write SetBorderColor;

    property Orientation: TIWMenuOrientation read FOrientation write SetOrientation;
  end;

  TIWMainMenuStyle = class(TIWMenuStyle);

  TIWSpaceItems = (itsNone, itsEvenlySpaced);

  TIWMenuAutoSize = (mnaNone, mnaFullWidth, mnaFullHeight);

  {$IFDEF VSNET}
  TMenuItemCollection = class;

  [TypeConverter('Atozed.Intraweb.Design.Controls.MenuItemConverter, Atozed.Intraweb.Design')]
  TMenuItem = class(MarshalByRefObject, ICloneable)
  protected
    FItems : TMenuItemCollection;
    FParent : TMenuItem;
    FEnabled : Boolean;
    FCaption : String;
    FImageIndex : Integer;
    FVisible : Boolean;
    function GetItem(Index: Integer): TMenuItem;
    function GetItems : TMenuItemCollection;
    function GetCount : Integer;
  public
    constructor Create; overload;
    constructor Create(AChildren : array of TMenuItem); overload;

    function Clone : &Object;
    procedure Remove;

    procedure Add(Item: TMenuItem);
    property Items[Index: Integer]: TMenuItem read GetItem; default;
    property ItemCollection : TMenuItemCollection read GetItems;
    property Count : Integer read GetCount;
  published
    function IsLine: Boolean;
    property ImageIndex: Integer read FImageIndex write FImageIndex;
    property Enabled: Boolean read FEnabled write FEnabled;
    property Caption: string read FCaption write FCaption;
    property Visible: Boolean read FVisible write FVisible;
  end;


  [Editor('Atozed.Intraweb.Design.Controls.MenuItemCollectionEditor, Atozed.Intraweb.Design', typeof(UITypeEditor))]
  TMenuItemCollection = class(ArrayList)
  protected
    FOwner : TMenuItem;
  public
    constructor Create(AOwner : TMenuItem);
    function Add(AItem : TMenuItem) : Integer;
    procedure AddRange(AItems : array of TMenuItem);
    procedure Remove(AItem : TMenuItem);
  end;

  TMainMenu = class
  public
    Items : TMenuItem;
  end;

  TIWMenuClickEvent = procedure(Sender: TObject; AItem: TMenuItem) of object;

  {$R icons\Atozed.Intraweb.TIWMenu.bmp}
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWMenu), 'TIWMenu.bmp')]
  {$ENDIF}
  TIWMenu = class(TIWCustomControl, IIWSubmitControl)
  private
    FComboBoxHide: Boolean;
  protected
    FSubmitParam : String;
    FAutoSize: TIWMenuAutoSize;
    FAttachedMenu: TMainMenu;
    FTimeOut: integer;
    FMenuLeft: integer;
    FMenuTop: integer;
    FPadding: integer;
    FTextOffset: integer;
    FItemSpacing: TIWSpaceItems;
    FFrameName: string;
    FMainMenuStyle: TIWMenuStyle;
    FSubMenuStyle: TIWMenuStyle;

    {$IFDEF VSNET}
    FOnItemClick : TIWMenuClickEvent;
    {$ENDIF}

    FMenuItems: TStringList;
    //
    procedure SetIWAutoSize(const Value: TIWMenuAutoSize);
    procedure SetMainMenuStyle(Value: TIWMenuStyle);
    procedure SetSubMenuStyle(Value: TIWMenuStyle);
    procedure SetAttachedMenu(Value: TMainMenu);
    procedure SetItemSpacing(Value: TIWSpaceItems);
    procedure SetPadding(Value: integer);
    procedure SetTextOffset(Value: integer);
    {$IFDEF VSNET}
    procedure InitControl; override;
    function GetItems : TMenuItemCollection;
    {$ENDIF}    
  protected
{$IFNDEF VSNET}
    procedure Notification(AComponent: {$IFDEF VSNET}
      TPlatformComponent{$ELSE}TComponent{$ENDIF};
      Operation: TOperation); override;
{$ENDIF}      

    procedure Submit(const AValue: string); override;
  public
    function GetSubmitParam : String;
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function RenderStyle(AContext: TIWBaseHTMLComponentContext): string; override;
    constructor Create(AOwner: TComponent); {$IFDEF VSNET}overload;{$ENDIF} override;
    {$IFDEF VSNET}
    constructor Create; overload;
    property AttachedMenu: TMainMenu read FAttachedMenu write SetAttachedMenu;
    {$ENDIF}
    destructor Destroy; override;

    property TextOffset: integer read FTextOffset;
  published
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    property Items : TMenuItemCollection read GetItems;
    property OnItemClick : TIWMenuClickEvent read FOnItemClick write FOnItemClick;
    {$ENDIF}
    property FriendlyName;
    {$IFNDEF VSNET}
    property AttachedMenu: TMainMenu read FAttachedMenu write SetAttachedMenu;
    {$ENDIF}
    property ItemSpacing: TIWSpaceItems read FItemSpacing write SetItemSpacing;
    property AutoSize: TIWMenuAutoSize read FAutoSize write SetIWAutoSize;
    {$IFDEF VSNET}[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]{$ENDIF}
    property MainMenuStyle: TIWMenuStyle read FMainMenuStyle write SetMainMenuStyle;
    {$IFDEF VSNET}[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]{$ENDIF}
    property SubMenuStyle: TIWMenuStyle read FSubMenuStyle write SetSubMenuStyle;
    property TimeOut: integer read FTimeOut write FTimeOut;
  private
    property ComboBoxHide: Boolean read FComboBoxHide write FComboBoxHide default true;
  end;
// Procs used during Paint

function FixCaption(Value: string): string;

implementation

uses
  {$IFNDEF VSNET}
    {$IFDEF Linux}QForms,{$ELSE}Forms, {$ENDIF}
  {$ENDIF}
  IWApplication, SysUtils, IWServer, IWForm, IWBaseHTMLControl, IWGlobal, IWTypes,
  IWLayoutMgrForm;

type
  TCharSet = set of AnsiChar;

function TrimCharRight(Value: string; Chars: TCharSet): string;
begin
  if Value <> '' then
  begin
    while (Length(Value) > 0) and (Value[Length(Value)] in Chars) do
      delete(Value, Length(Value), 1);
  end;
  Result := Value;
end;

function TrimCharLeft(Value: string; Chars: TCharSet): string;
begin
  if Length(Value) > 0 then
  begin
    while (Length(Value) > 0) and (Value[1] in Chars) do
      delete(Value, 1, 1);
  end;
  Result := Value;
end;

function FixCaption(Value: string): string;
var
  i: integer;
begin
  Result := '';
  for i := 1 to length(value) do
  begin
    if value[i] <> '&' then
      Result := Result + Value[i];
  end;
end;

//
//	Make sure that the double quotes (") are correctly formatted
//

function DQuoteStr(const Value: string): string;
begin
  Result := '"' + TrimCharRight(TrimCharLeft(Value, ['''', '"']), ['''', '"']) + '"';
end;

{ TIWMenuStyle }

constructor TIWMenuStyle.Create;
begin
  inherited Create;
  // Set the Font defaults
  FIWMenu := AIWMenu;
  FSelectedMenuItemFont := TIWFont.Create;
  FMenuItemFont := TIWFont.Create;
  FDisabledMenuItemFont := TIWFont.Create;

  FBGColor := fromTColor(clNone);
  FItemBGColor := fromTColor(clNone);
  FSelectedItemBGColor := fromTColor(clNone);
  FDisabledItemBGColor := fromTColor(clNone);
  FBorderColor := fromTColor(clNone);

  FDisabledMenuItemFont.Color := fromTColor(clLtGray);
end;

destructor TIWMenuStyle.Destroy;
begin
  FreeAndNil(FDisabledMenuItemFont);
  FreeAndNil(FSelectedMenuItemFont);
  FreeAndNil(FMenuItemFont);

  inherited Destroy;
end;

procedure TIWMenuStyle.SetOrientation(const Value: TIWMenuOrientation);
begin
  FOrientation := Value;
  FIWMenu.Invalidate;
end;

procedure TIWMenuStyle.SetMenuItemFont(const Value: TIWFont);
begin
  FMenuItemFont.Assign(Value);
  FIWMenu.Invalidate;
end;

procedure TIWMenuStyle.SetSelectedMenuItemFont(const Value: TIWFont);
begin
  FSelectedMenuItemFont.Assign(Value);
  FIWMenu.Invalidate;
end;

procedure TIWMenuStyle.SetDisabledMenuItemFont(const Value: TIWFont);
begin
  FDisabledMenuItemFont.Assign(Value);
  FIWMenu.Invalidate;
end;

procedure TIWMenuStyle.Assign(Source: TPersistent);
begin
  if Source is TIWMenuStyle then begin
    with Source as TIWMenuStyle do begin
      Self.FSelectedMenuItemFont.Assign(SelectedMenuItemFont);
      Self.FColor := FColor;
      Self.FMenuItemFont.Assign(FMenuItemFont);
      Self.FOrientation := FOrientation;
      Self.FIWMenu := FIWMenu;
      Self.FDisabledMenuItemFont.Assign(FDisabledMenuItemFont);
    end;
  end
  else
    inherited Assign(Source);
end;

procedure TIWMenuStyle.SetDisabledItemBGColor(const Value: TIWColor);
begin
  FDisabledItemBGColor := Value;
  FIWMenu.Invalidate;
end;

procedure TIWMenuStyle.SetItemBGColor(const Value: TIWColor);
begin
  FItemBGColor := Value;
  FIWMenu.Invalidate;
end;

procedure TIWMenuStyle.SetSelectedItemBGColor(const Value: TIWColor);
begin
  FSelectedItemBGColor := Value;
  FIWMenu.Invalidate;
end;

procedure TIWMenuStyle.SetBGColor(const Value: TIWColor);
begin
  FBGColor := Value;
  FIWMenu.Invalidate;
end;

procedure TIWMenuStyle.SetBorderColor(const Value: TIWColor);
begin
  FBorderColor := Value;
  FIWMenu.Invalidate; 
end;

{ TIWMenu }

constructor TIWMenu.Create(AOwner: TComponent);
begin
{$IFDEF VSNET}
  inherited Create;
{$ELSE}
  inherited Create(AOwner);

  FMainMenuStyle := TIWMainMenuStyle.Create(self);

  //#e0dfe3
  FMainMenuStyle.BGColor := fromTColor( TColor($e3dfe0) );

  FSubMenuStyle := TIWMenuStyle.Create(self);

  FSubMenuStyle.Orientation := iwOVertical;

  FAutoSize := mnaNone;
  Height := 20;
  Width := 200;
  FTextOffset := 30;
  ItemSpacing := itsNone;

  FMenuItems := TStringList.Create;
  ComboBoxHide := true;

  SupportsPartial := false;
{$ENDIF}
end;

destructor TIWMenu.Destroy;
begin
  FreeAndNil(FMainMenuStyle);
  FreeAndNil(FSubMenuStyle);
  FreeAndNil(FMenuItems);
  inherited;
end;

procedure TIWMenu.SetMainMenuStyle(Value: TIWMenuStyle);
begin
  FMainMenuStyle.Assign(Value);
  Invalidate;
end;

procedure TIWMenu.SetSubMenuStyle(Value: TIWMenuStyle);
begin
  FSubMenuStyle.Assign(Value);
  Invalidate;
end;

procedure TIWMenu.SetAttachedMenu(Value: TMainMenu);
begin
  FAttachedMenu := Value;
  Invalidate;
end;

procedure TIWMenu.SetItemSpacing(Value: TIWSpaceItems);
begin
  if FItemSpacing <> Value then
  begin
    FItemSpacing := Value;
    if FItemSpacing = itsNone then
      FAutoSize := mnaNone;

    Invalidate;
  end;
end;

procedure TIWMenu.SetPadding(Value: integer);
begin
  if FPadding <> Value then
  begin
    FPadding := Value;
    Invalidate;
  end;
end;

procedure TIWMenu.SetTextOffset(Value: integer);
begin
  if FTextOffset <> Value then
  begin
    FTextOffset := Value;
    Invalidate;
  end;
end;

//
//	THIS IS THE JAVASCRIPT/HTML VERSION OF THE CODE
//

Type
  TIWHTMLTable = class(TIWHTMLTag)
  protected
    FIWMenu: TIWMenu;
    FMenuItem: TMenuItem;
  public
    constructor Create(AMenu: TIWMenu; AMenuItem: TMenuItem); reintroduce;  
    function generateMenu(AContext: TIWComponent40Context; emptyElement: TIWHTMLTag; menuStyle: TIWMenuStyle; AIdent: Integer = 0): String;
    function generateMenuElement(AContext: TIWComponent40Context; AMenuItem: TMenuItem; menuStyle: TIWMenuStyle): TIWHTMLTag;
  end;

function TIWMenu.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LScript: String;
  LMainMenuStyle: TIWMenuStyle;
begin
  Result := nil;
  set_RenderSize(AContext.ContainerContext.LayoutManager is TIWLayoutMgrForm); 
  if Assigned(FAttachedMenu) and (FAttachedMenu.Items.Count > 0) then begin

    (AContext as TIWComponent40Context).AddScriptFile('/js/IWMenu.js');

    // Result := TIWHTMLTag.CreateTag(''); try
      with (AContext.PageContext as TIWPageContext40).StyleTag do begin

        with MainMenuStyle.FSelectedMenuItemFont do
          if (Trim(CSSStyle) = '') or (Trim(CSSStyle) = 'MainMenuSelectedItemStyle') then begin
            CSSStyle := '';
            Contents.AddText('.MainMenuSelectedItemStyle {');
            Contents.AddText('  font-size: 12;font-family: arial;');
            Contents.AddText('  border: 1px solid ' + iif(MainMenuStyle.BorderColor <> fromTColor(clNone),
                                                          ColorToRGBString(MainMenuStyle.BorderColor),
                                                          '#9d9da1') + ';');
            Contents.AddText('  z-index: ' + IntToStr(ZIndex + 100)+';');
            Contents.AddText(FontToStringStyle(AContext.Browser));
            if MainMenuStyle.SelectedItemBGColor <> fromTColor(clNone) then begin
              Contents.AddText('  background-color:' + ColorToRGBString(MainMenuStyle.SelectedItemBGColor) + ';');
            end else begin
              Contents.AddText('  background-color:#b2b4bf;');
            end;
            Contents.AddText('}' + EOL);
            CSSStyle := 'MainMenuSelectedItemStyle';
          end;

        with MainMenuStyle.FMenuItemFont do
          if (Trim(CSSStyle) = '') or (Trim(CSSStyle) = 'MainMenuItemStyle') then begin
            CSSStyle := '';
            Contents.AddText('.MainMenuItemStyle {');
            Contents.AddText('  font-size: 12;font-family: arial; text-align:left;');
            Contents.AddText('  z-index: ' + IntToStr(ZIndex + 100)+';');
            Contents.AddText(FontToStringStyle(AContext.Browser));
            if MainMenuStyle.ItemBGColor <> fromTColor(clNone) then begin
              Contents.AddText('  background-color:' + ColorToRGBString(MainMenuStyle.ItemBGColor) + ';');
            end else begin
              Contents.AddText('background-color:#e0dfe3;');  
            end;
            Contents.AddText('  border: 1px solid ' + iif(MainMenuStyle.BorderColor <> fromTColor(clNone),
                                                          ColorToRGBString(MainMenuStyle.BorderColor),
                                                          '#e0dfe3') + ';');
            Contents.AddText('}' + EOL);
            CSSStyle := 'MainMenuItemStyle';
          end;

        with MainMenuStyle.FDisabledMenuItemFont do
          if (Trim(CSSStyle) = '') or (Trim(CSSStyle) = 'MainDisabledMenuItemStyle') then begin
            CSSStyle := '';
            Contents.AddText('.MainDisabledMenuItemStyle {');
            Contents.AddText('  text-align: left;');
            Contents.AddText('  z-index: ' + IntToStr(ZIndex + 100)+';');
            Contents.AddText(FontToStringStyle(AContext.Browser));
            if MainMenuStyle.DisabledItemBGColor <> fromTColor(clNone) then
              Contents.AddText('  background-color:' + ColorToRGBString(MainMenuStyle.DisabledItemBGColor) + ';');
            Contents.AddText('  border: 1px solid ' + iif(MainMenuStyle.BorderColor <> fromTColor(clNone),
                                                          ColorToRGBString(MainMenuStyle.BorderColor),
                                                          '#FFFFFF') + ';');
            Contents.AddText('}' + EOL);
            CSSStyle := 'MainDisabledMenuItemStyle';
          end;

        Contents.AddText('.MenuStyle {');
        Contents.AddText('  text-align: left;');
        if SubMenuStyle.BGColor <> fromTColor(clNone) then
          Contents.AddText('  background-color:' + ColorToRGBString(SubMenuStyle.BGColor) + ';');

        Contents.AddText('  border: 1px solid ' + iif(SubMenuStyle.BorderColor <> fromTColor(clNone),
                                                      ColorToRGBString(SubMenuStyle.BorderColor),
                                                      '#FFFF') + ';');
        Contents.AddText('  z-index: ' + IntToStr(ZIndex + 100)+';');
        Contents.AddText('}' + EOL);

        with SubMenuStyle.FSelectedMenuItemFont do
          if (Trim(CSSStyle) = '') or (Trim(CSSStyle) = 'MenuSelectedItemStyle') then begin
            CSSStyle := '';
            Contents.AddText('.MenuSelectedItemStyle {');
            Contents.AddText('  font-size: 12;font-family: arial; text-align:left;');
            Contents.AddText('  z-index: ' + IntToStr(ZIndex + 100)+';');
            if SubMenuStyle.SelectedItemBGColor <> fromTColor(clNone) then
              Contents.AddText('  background-color:' + ColorToRGBString(SubMenuStyle.SelectedItemBGColor) + ';')
            else
              Contents.AddText('  background-color:#b2b4bf;');

            Contents.AddText(FontToStringStyle(AContext.Browser));
            Contents.AddText('  border: 1px solid ' + iif(SubMenuStyle.BorderColor <> fromTColor(clNone),
                                                          ColorToRGBString(SubMenuStyle.BorderColor),
                                                          '#9d9da1') + ';');
            Contents.AddText('}' + EOL);
            CSSStyle := 'MenuSelectedItemStyle';
          end;

        with SubMenuStyle.FMenuItemFont do
          if (Trim(CSSStyle) = '') or (Trim(CSSStyle) = 'MenuItemStyle') then begin
            CSSStyle := '';
            Contents.AddText('.MenuItemStyle {');
            Contents.AddText('  font-size: 12;font-family: arial; text-align:left;');
            Contents.AddText('  z-index: ' + IntToStr(ZIndex + 100)+';');
            if SubMenuStyle.ItemBGColor <> fromTColor(clNone) then
              Contents.AddText('  background-color:' + ColorToRGBString(SubMenuStyle.ItemBGColor) + ';')
            else
              Contents.AddText('  background-color:#FFFFFF;');

            Contents.AddText(FontToStringStyle(AContext.Browser));
            Contents.AddText('  border: 1px solid ' + iif(SubMenuStyle.BorderColor <> fromTColor(clNone),
                                                          ColorToRGBString(SubMenuStyle.BorderColor),
                                                          '#FFFFFF') + ';');
            Contents.AddText('}' + EOL);
            CSSStyle := 'MenuItemStyle';
          end;

        with SubMenuStyle.FDisabledMenuItemFont do
          if (Trim(CSSStyle) = '') or (Trim(CSSStyle) = 'DisabledMenuItemStyle') then begin
            CSSStyle := '';
            Contents.AddText('.DisabledMenuItemStyle {');
            Contents.AddText('  text-align: left;');
            Contents.AddText('  z-index: ' + IntToStr(ZIndex + 100)+';');
            if SubMenuStyle.DisabledItemBGColor <> fromTColor(clNone) then
              Contents.AddText('  background-color:' + ColorToRGBString(SubMenuStyle.DisabledItemBGColor) + ';');
            Contents.AddText(FontToStringStyle(AContext.Browser));
            Contents.AddText('  border: 1px solid ' + iif(SubMenuStyle.BorderColor <> fromTColor(clNone),
                                                          ColorToRGBString(SubMenuStyle.BorderColor),
                                                          '#FFFFFF') + ';');
            Contents.AddText('}' + EOL);
            CSSStyle := 'DisabledMenuItemStyle';
          end;
      end;

      result := TIWHTMLTag.CreateTag(''); try
      // result := TIWHTMLTable.Create(self, AttachedMenu.Items); try;
      LMainMenuStyle := TIWMainMenuStyle.Create(self); try
        LMainMenuStyle.Assign(MainMenuStyle);
        FMenuItems.Clear;

        with result do begin // result.Contents.AddTag('div') do begin
        //  AddStringParam('id', HTMLName);
        //  AddStringParam('name', HTMLName);
        //  AddStringParam('class', HTMLName + 'CSS');

          FMenuItems.AddObject('', AttachedMenu.Items);
          // with TIWHTMLTable(result) do begin
          with TIWHTMLTable(Contents.AddTagAsObject(TIWHTMLTable.Create(self, AttachedMenu.Items))) do begin // First level menu
            // AddStringParam('width', '100%');
            // AddStringParam('height', '100%');
            LScript := generateMenu(TIWComponent40Context(AContext), result, LMainMenuStyle);
            AddStringParam('style', RenderStyle(AContext));
            // AddStringParam('class', HTMLName + 'CSS');
            AddStringParam('id', HTMLName);
          end;
        end;

        with TIWComponent40Context(AContext) do begin
          AddToJavaScriptOnce('var ' + HTMLName + '_menu;');

          LScript := HTMLName + '_menu = initMenu("' + HTMLName + {'_menu}'")' + EOL
            + 'with (' + HTMLName + '_menu) {' + EOL
            + '  SelectedItemStyle = "' + LMainMenuStyle.SelectedMenuItemFont.CSSStyle + '";' + EOL
            + '  MenuOrientation = ' +
              iif(LMainMenuStyle.Orientation = iwOHorizontal, 'moHorizontal',
              iif(LMainMenuStyle.Orientation = iwOVertical, 'moVertical', 'error'))
            + ';' + EOL
            + lScript + '}' + EOL;

          LScript := LScript + HTMLName + '_menu.ComboBoxHide = ' + iif(FComboBoxHide, 'true', 'false') + ';';

          CacheImage(HTMLName + '_arrow', WebApplication.InternalURLBase + '/gfx/menu_arrow_right.gif');
        end;
      finally
        LMainMenuStyle.Free;
      end;

      TIWComponent40Context(AContext).AddToInitProc(LScript);
    except FreeAndNil(Result); raise; end;
  end;
end;

function TIWMenu.RenderStyle(AContext: TIWBaseHTMLComponentContext): string;
begin
 result := inherited RenderStyle(AContext);
 result := result + iif(MainMenuStyle.BGColor <> fromTColor(clNone), '  background-color:' + ColorToRGBString(MainMenuStyle.BGColor) + ';');
end;

{$IFNDEF VSNET}
procedure TIWMenu.Notification(AComponent:
  {$IFDEF VSNET}TPlatformComponent{$ELSE}TComponent{$ENDIF};
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then begin
    if AComponent = FAttachedMenu then
    begin
      FAttachedMenu := nil;
      Invalidate;
    end;
  end;
end;
{$ENDIF}

procedure TIWMenu.SetIWAutoSize(const Value: TIWMenuAutoSize);
begin
  FAutoSize := Value;
  if Value <> mnaNone then
    ItemSpacing := itsEvenlySpaced;
  Invalidate;
end;

procedure TIWMenu.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
  {$IFDEF VSNET}
  if Assigned(FOnItemClick) then begin
    FOnItemClick( Self,TMenuItem(FMenuItems.Objects[FMenuItems.IndexOf(AValue)]) );
  end;
  {$ELSE}
  TMenuItem(FMenuItems.Objects[FMenuItems.IndexOf(AValue)]).Click;
  {$ENDIF}
end;

function TIWMenu.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

{ TIWHTMLTable }

constructor TIWHTMLTable.Create(AMenu: TIWMenu; AMenuItem: TMenuItem);
begin
  inherited CreateTag('table');
  FIWMenu := AMenu;
  FMenuItem := AMenuItem;
end;

function TIWHTMLTable.generateMenuElement(AContext: TIWComponent40Context; AMenuItem: TMenuItem; menuStyle: TIWMenuStyle): TIWHTMLTag;
Var
{$IFNDEF VSNET}
  lImgList: TIWImageList;
{$ENDIF}
  LPathName: String;
  LCursor: String;
begin
  if AMenuItem.Enabled then begin
    AddStringParam('class', menuStyle.FMenuItemFont.CSSStyle);
  end else begin
    AddStringParam('class', menuStyle.FDisabledMenuItemFont.CSSStyle);
  end;
  AddIntegerParam('border', 0);
  AddStringParam('width', '100%');
  AddStringParam('height', '100%');
  AddIntegerParam('cellspacing', 0);
  AddIntegerParam('cellpadding', 0);

  case {$IFDEF CLR}FIWMenu.WebCursor{$ELSE}FIWMenu.Cursor{$ENDIF} of
    crCrossHair: LCursor := 'crosshair;';
    crDefault: LCursor := 'default;';
    crPointer: LCursor := 'pointer;';
    crMove: LCursor := 'move;';
    crEResize: LCursor := 'e-resize;';
    crNEResize: LCursor := 'ne-resize;';
    crNWResize: LCursor := 'nw-resize;';
    crNResize: LCursor := 'n-resize;';
    crSEResize: LCursor := 'se-resize;';
    crSWResize: LCursor := 'sw-resize;';
    crSResize: LCursor := 's-resize;';
    crWResize: LCursor := 'w-resize;';
    crText: LCursor := 'text;';
    crWait: LCursor := 'wait;';
    crHelp: LCursor := 'help;';
  end;

  if LCursor <> '' then begin
    AddStringParam('style', 'cursor:' + LCursor);
  end;

  with FIWMenu do begin
    AddStringParam('id', HTMLName + '_submenu' + IntToStr(FMenuItems.IndexOfObject(AMenuItem)));
    FMenuItems[FMenuItems.IndexOfObject(AMenuItem)] :=
      HTMLName + '_submenu' + IntToStr(FMenuItems.IndexOfObject(AMenuItem));
  end;

  case menuStyle.Orientation of
    iwoHorizontal: begin
      with Contents.AddTag('tr') do begin
        with Contents.addtag('td') do begin
          AddStringParam('width', '16px');
          Contents.AddText('&nbsp;');
        end;
        
        with contents.AddTag('td') do begin
          Add('nowrap');
          Contents.AddText(StringReplace(AMenuItem.Caption, '&', '', [rfReplaceall]));
        end;

        with Contents.addtag('td') do begin
          AddStringParam('width', '16px');
          Contents.AddText('&nbsp;');
        end;
      end;
    end;
    iwOVertical: begin
      with contents.AddTag('tr') do begin
        with Contents.AddTag('td') do begin
          AddStringParam('width', '24px');
          AddStringParam('align', 'left');
          AddStringParam('valign', 'middle');
          // AddStringParam('class', menuprop.getMenuItem().getCSSStyle());
          {$IFNDEF VSNET}
          if (FIWMenu.AttachedMenu.Images <> nil) and (AMenuItem.ImageIndex > -1) then begin
            lImgList := TIWImageList(FIWMenu.AttachedMenu.Images);
            LPathname := lImgList.ExtractImageToCache(AMenuItem.ImageIndex);
            if LPathname <> '' then begin
              with Contents.AddTag('img') do begin
                AddStringParam('src', GServerController.CacheURL + LPathname);
                AddIntegerParam('border', 0);
              end;
            end else begin
              contents.AddText('&nbsp;');
            end;
          end else begin
            contents.AddText('&nbsp;');
          end;
          {$ELSE}
            contents.AddText('&nbsp;');
          {$ENDIF}
        end;

        with Contents.AddTag('td') do begin
          contents.AddText(StringReplace(AMenuItem.Caption, '&', '', [rfReplaceall]))
        end;

        with Contents.AddTag('td') do begin
          AddStringParam('width', '8px');
          contents.AddText('&nbsp;');
        end;

        (* with Contents.AddTag('td') do begin
          // ShortCutToText(AMenuItem.ShortCut)
          AddStringParam('align', 'right');
          contents.AddText(StringReplace(ShortCutToText(AMenuItem.ShortCut), '&', '', [rfReplaceall]))
        end; *)

        with Contents.AddTag('td') do begin
          AddStringParam('width', '16px');
          if AMenuItem.Count > 0 then begin
            with contents.AddTag('img') do begin
              AddStringParam('src', AContext.WebApplication.InternalURLBase + '/gfx/menu_arrow_right.gif');
            end;
          end else begin
            contents.AddText('&nbsp;')
          end;
        end;
      end;
    end;
  end;
  result := self;
end;

function TIWHTMLTable.generateMenu(AContext: TIWComponent40Context; emptyElement: TIWHTMLTag;
  menuStyle: TIWMenuStyle; AIdent: Integer = 0): String;
Var
  i: Integer;
  LIdent: String;
  tr: TIWHTMLTag;
  lMenuStyle: TIWMenuStyle;
begin
  result := '';
  tr := nil;

  // do_onMenuRender(m_Menu, context, newProp);

  AddStringParam('id', FIWMenu.HTMLName + '_submenu' + IntToStr(FIWMenu.FMenuItems.IndexOfObject(FMenuItem)) + '_table');
  FIWMenu.FMenuItems[FIWMenu.FMenuItems.IndexOfObject(FMenuItem)] :=
    FIWMenu.HTMLName + '_submenu' + IntToStr(FIWMenu.FMenuItems.IndexOfObject(FMenuItem)) + '_table';
  FIWMenu.FMenuItems[FIWMenu.FMenuItems.IndexOfObject(FMenuItem)] :=
    FIWMenu.HTMLName + '_submenu' + IntToStr(FIWMenu.FMenuItems.IndexOfObject(FMenuItem));
  if not (menuStyle is TIWMainMenuStyle) then begin
    AddStringParam('class', 'MenuStyle');
  end;
  AddIntegerParam('border', 0);
  AddIntegerParam('cellspacing', 0);
  AddIntegerParam('cellpadding', 0);
  AddStringParam('style', 'position:absolute; visibility:hidden; display:none; z-index:' + IntToStr(FIWMenu.ZIndex + 100));

  for i := 0 to FMenuItem.Count - 1 do begin
      if FMenuItem[i].Visible then begin
      if FMenuItem[i].IsLine then begin
        // the item is separator
        if menuStyle.Orientation = iwoVertical then begin
          Contents.AddTag('tr').Contents.AddTag('td').Contents.AddTag('hr');
        end;
      end else begin
        // the item is NOT separator
        FIWMenu.FMenuItems.AddObject('', FMenuItem[i]);
        if menuStyle.Orientation = iwoVertical then begin
          with contents.AddTag('tr').Contents.addtag('td') do
            TIWHTMLTable(Contents.AddTagAsObject(
              TIWHTMLTable.Create(FIWMenu, FMenuItem))).generateMenuElement(AContext, FMenuItem[i], menuStyle);
        end else begin
          if not assigned(tr) then begin
            tr := TIWHTMLTag.CreateTag('tr');
            contents.AddTagAsObject(tr);
          end;
          with tr.Contents.addtag('td') do
            TIWHTMLTable(Contents.AddTagAsObject(
              TIWHTMLTable.Create(FIWMenu, FMenuItem))).generateMenuElement(AContext, FMenuItem[i], menuStyle);
        end;
      end;
    end;
  end;

  if (FMenuItem = FIWMenu.AttachedMenu.Items) and (FIWMenu.FItemSpacing = itsNone) then begin
    if menuStyle.Orientation = iwoVertical then begin
      with contents.AddTag('tr').Contents.addtag('td') do begin
        AddStringParam('height', '100%');
        Contents.AddText('&nbsp;');
      end;
    end else begin
      if not assigned(tr) then begin
        tr := TIWHTMLTag.CreateTag('tr');
        contents.AddTagAsObject(tr);
      end;
      with tr.Contents.addtag('td') do begin
        AddStringParam('width', '100%');
        Contents.AddText('&nbsp;');
      end;
    end;
  end;

  LIdent := StringOfChar(' ', AIdent * 2);

  for i := 0 to FMenuItem.Count - 1 do begin
    if FMenuItem[i].Visible then begin
      if not FMenuItem[i].IsLine then begin
        Result := Result + LIdent + 'with (addMenuItem("' + FIWMenu.HTMLName + '_submenu'
              + IntToStr(FIWMenu.FMenuItems.IndexOfObject(FMenuItem[i])) + '", id)) { ' + EOL
              + LIdent + '  SelectedItemStyle = "' + menuStyle.SelectedMenuItemFont.CSSStyle + '";'
              + EOL;

        if FMenuItem[i].Count > 0 then begin
          with TIWHTMLTable(emptyElement.Contents.AddTagAsObject(TIWHTMLTable.Create(FIWMenu, FMenuItem[i]))) do begin
            lMenuStyle := TIWMenuStyle.Create(FIWMenu); try
              lMenuStyle.Assign(FIWMenu.FSubMenuStyle);
              result := result + generateMenu(AContext, emptyElement, lMenuStyle, AIdent + 1);
              result := result + LIdent + '  MenuOrientation = '
                + iif(lMenuStyle.Orientation = iwOHorizontal, 'moHorizontal',
                  iif(lMenuStyle.Orientation = iwOVertical, 'moVertical', 'error'))
                +  ';' + EOL;
            finally
              lMenuStyle.Free;
            end;
          end;
        end;

        result := result + LIdent + '}' + EOL;
      end;
    end;
  end;
end;

{$IFDEF VSNET}
constructor TMenuItem.Create;
begin
  Inherited;
  FItems := TMenuItemCollection.Create(Self);
  FEnabled := True;
  FVisible := True;
end;

constructor TMenuItem.Create(AChildren : array of TMenuItem);
begin
  Create;

  FItems.AddRange(AChildren);
end;

procedure TMenuItem.Remove;
begin
  (FParent.FItems as IList).Remove(Self);
  FParent := nil;
end;

// ICloneable Member
function TMenuItem.Clone : &Object;
var
  LItem : TMenuItem;
  i : Integer;
  LChild : TMenuItem;
begin
  LItem := Activator.CreateInstance( Self.GetType() ) as TMenuItem;

  // copy properties
  LItem.FEnabled := FEnabled;
  LItem.FCaption := FCaption;
  LItem.FImageIndex := FImageIndex;
  LItem.FVisible := FVisible;

  // deep copy the children
  for i := 0 to FItems.Count - 1 do begin
    LChild := FItems[i] as TMenuItem;
    LItem.FItems.Add( LChild.Clone() as TMenuItem);
  end;

  result := LItem;
end;

function TMenuItem.GetItems : TMenuItemCollection;
begin
  Result := FItems;
end;

function TMenuItem.IsLine: Boolean;
begin
  Result := FCaption = '-';
end;

function TMenuItem.GetItem(Index: Integer): TMenuItem;
begin
  Result := TMenuItem(FItems[Index]);
end;

function TMenuItem.GetCount : Integer;
begin
  Result := FItems.Count;
end;

procedure TMenuItem.Add(Item: TMenuItem);
begin
  FItems.Add(Item);
end;

constructor TIWMenu.Create;
begin
  Inherited;
end;

procedure TIWMenu.InitControl;
begin
  Inherited;
  FMainMenuStyle := TIWMainMenuStyle.Create(self);
  FSubMenuStyle := TIWMenuStyle.Create(self);

  FSubMenuStyle.Orientation := iwOVertical;

  FAutoSize := mnaNone;
  Height := 20;
  Width := 200;
  FTextOffset := 30;
  ItemSpacing := itsNone;

  FMenuItems := TStringList.Create;
  ComboBoxHide := true;

  SupportsPartial := false;

  FAttachedMenu := TMainMenu.Create;
  FAttachedMenu.Items := TMenuItem.Create;
end;

function TIWMenu.GetItems : TMenuItemCollection;
begin
  Result := FAttachedMenu.Items.ItemCollection;
end;

{ TMenuItemCollection }

constructor TMenuItemCollection.Create(AOwner : TMenuItem);
begin
  inherited Create;
  FOwner := AOwner;
end;

function TMenuItemCollection.Add(AItem : TMenuItem) : Integer;
var
  LItemAsObject : &Object;
begin
  AItem.FParent := Self.FOwner;
  LItemAsObject := AItem As &Object;
  result := inherited Add(LItemAsObject);
end;

procedure TMenuItemCollection.AddRange(AItems : array of TMenuItem);
var
  i,l : Integer;
begin
  l := Length(AItems);
  for i := 0 to l - 1 do begin
    Add( AItems[i] );
  end;

end;

procedure TMenuItemCollection.Remove(AItem : TMenuItem);
begin
  AItem.Remove;
end;

{$ENDIF}


initialization
  TIWServer.AddInternalFile('IW_JS_IWMENU', '/js/IWMenu.js');
  TIWServer.AddInternalFile('IW_GFX_menu_arrow_right', '/gfx/menu_arrow_right.gif');
end.



